"""
Proxy Manager Utility
Manages proxy list in proxies.txt file
"""
import os
import random
from typing import List, Optional


PROXIES_FILE = "proxies.txt"


def read_proxies() -> List[str]:
    """
    Read proxies from proxies.txt file.
    
    Returns:
        List of proxy strings
    """
    proxies = []
    try:
        if os.path.exists(PROXIES_FILE):
            with open(PROXIES_FILE, 'r', encoding='utf-8') as file:
                for line in file:
                    line = line.strip()
                    if line and not line.startswith('#'):
                        proxies.append(line)
    except Exception as e:
        print(f"Error reading proxies file: {e}")
    return proxies


def write_proxies(proxies: List[str]) -> bool:
    """
    Write proxies to proxies.txt file.
    
    Args:
        proxies: List of proxy strings
        
    Returns:
        True if successful, False otherwise
    """
    try:
        with open(PROXIES_FILE, 'w', encoding='utf-8') as file:
            for proxy in proxies:
                if proxy.strip():
                    file.write(f"{proxy.strip()}\n")
        return True
    except Exception as e:
        print(f"Error writing proxies file: {e}")
        return False


def add_proxy(proxy: str) -> bool:
    """
    Add a proxy to the proxies.txt file.
    
    Args:
        proxy: Proxy string to add
        
    Returns:
        True if successful, False otherwise
    """
    proxies = read_proxies()
    proxy = proxy.strip()
    if proxy and proxy not in proxies:
        proxies.append(proxy)
        return write_proxies(proxies)
    return False


def remove_proxy(proxy: str) -> bool:
    """
    Remove a proxy from the proxies.txt file.
    
    Args:
        proxy: Proxy string to remove
        
    Returns:
        True if successful, False otherwise
    """
    proxies = read_proxies()
    proxy = proxy.strip()
    if proxy in proxies:
        proxies.remove(proxy)
        return write_proxies(proxies)
    return False


def get_random_proxy() -> Optional[str]:
    """
    Get a random proxy from the proxies.txt file.
    
    Returns:
        Random proxy string or None if no proxies available
    """
    proxies = read_proxies()
    if proxies:
        return random.choice(proxies)
    return None


def get_proxy_url(proxy: str) -> str:
    """
    Convert proxy string to full proxy URL.
    
    Args:
        proxy: Proxy string (e.g., "username:password@host:port")
        
    Returns:
        Full proxy URL with http:// prefix
    """
    if proxy.startswith('http://') or proxy.startswith('https://'):
        return proxy
    return f"http://{proxy}"


def get_proxy_dict(proxy: Optional[str] = None) -> dict:
    """
    Get proxy dictionary for requests library.
    
    Args:
        proxy: Optional proxy string. If None, uses random proxy from file.
        
    Returns:
        Dictionary with http and https proxy URLs
    """
    if proxy is None:
        proxy = get_random_proxy()
    
    if proxy is None:
        return {}
    
    proxy_url = get_proxy_url(proxy)
    return {
        "http": proxy_url,
        "https": proxy_url
    }

