import csv
import json
import os
import random
import time
import logging
import threading
import uuid
import asyncio
import imaplib
import email
import re
import tempfile
import string
from concurrent.futures import ThreadPoolExecutor, as_completed
from typing import Dict, List, Optional, Set, Tuple
from datetime import datetime

import pyotp
from fastapi import APIRouter, HTTPException, File, UploadFile, Form, Body
from pydantic import BaseModel, validator
from instagrapi import Client
from instagrapi.mixins.challenge import ChallengeChoice
from dotenv import load_dotenv
from supabase import create_client
from app.task_manager import background_tasks

load_dotenv()

SESSIONS_DIR = os.path.join(os.path.dirname(os.path.dirname(__file__)), "sessions")
VERIFY_CSV = os.path.join(os.path.dirname(os.path.dirname(__file__)), "verify.csv")
PROXIES_TXT = os.path.join(os.path.dirname(os.path.dirname(__file__)), "proxies.txt")
DEFAULT_LOCALE = "en_US"
DEFAULT_TZ_OFFSET = -7 * 60 * 60  # Los Angeles by default
# Proxy configuration is now loaded from proxies.txt file
NOACTIVE_CSV = os.path.join(os.path.dirname(os.path.dirname(__file__)), "noactive account.csv")
MAX_WORKERS = 5
DEFAULT_IMAP_PASSWORD = "Heritech@098A!2"
DEFAULT_IMAP_PORT = 993

# Cache for loaded proxies
_proxies_cache: Optional[List[str]] = None

# Logger setup - prevent duplicate logs
logger = logging.getLogger("instgrapi_app")
logger.setLevel(logging.INFO)
# Prevent propagation to root logger to avoid duplicate logs
logger.propagate = False
# Only add handler if it doesn't exist
if not logger.handlers:
    handler = logging.StreamHandler()
    handler.setLevel(logging.INFO)
    formatter = logging.Formatter("%(asctime)s - %(levelname)s - %(message)s")
    handler.setFormatter(formatter)
    logger.addHandler(handler)

# Initialize Supabase client
SUPABASE_URL = os.getenv("SUPABASE_URL")
SUPABASE_KEY = os.getenv("SUPABASE_KEY")
supabase = None
if SUPABASE_URL and SUPABASE_KEY:
    try:
        supabase = create_client(SUPABASE_URL, SUPABASE_KEY)
        logger.info("✅ Supabase client initialized")
    except Exception as e:
        logger.warning(f"⚠️ Failed to initialize Supabase client: {e}")
else:
    logger.warning("⚠️ Supabase credentials not found in environment variables")


# CSV Generation Functions
def generate_random_id(length=15) -> str:
    """Generate random ID like '01fmcQ5wGAyjbDqB'"""
    chars = string.ascii_letters + string.digits
    return ''.join(random.choices(chars, k=length))


def generate_workflow_id(length=15) -> str:
    """Generate random workflow ID like 'kGaq1E86JkRca3Rg'"""
    chars = string.ascii_letters + string.digits
    return ''.join(random.choices(chars, k=length))


def generate_account_string(username: str, password: str, session_data: Dict, user_agent: str, cookies: str, proxy: Optional[str] = None, email: Optional[str] = None, email_password: Optional[str] = None) -> str:
    """
    Generate account_string format:
    username:password|Instagram version|User-Agent|cookies|proxy|email:email_password
    """
    
    # Extract Instagram version from device_settings
    instagram_version = session_data.get("device_settings", {}).get("app_version", "385.0.0.47.74")
    android_version = session_data.get("device_settings", {}).get("android_version", "26")
    android_release = session_data.get("device_settings", {}).get("android_release", "8.0")
    
    # Format: Instagram version Android (version/release; dpi; resolution; manufacturer; device; model; cpu; locale; version_code)
    instagram_info = f"Instagram {instagram_version} Android ({android_version}/{android_release}; 560dpi; 1440x2560; samsung; SM-G930F; herolte; samsungexynos8890; en_US; 750732795)"
    
    # Extract UUIDs for device identifiers
    uuids = session_data.get("uuids", {})
    android_device_id = uuids.get("android_device_id", "android-f1b15afc37b59351")
    phone_id = uuids.get("phone_id", "")
    uuid_val = uuids.get("uuid", "")
    advertising_id = uuids.get("advertising_id", "")
    
    # Build UUID string (separated by semicolons)
    uuid_parts = [android_device_id]
    if phone_id:
        uuid_parts.append(phone_id)
    if uuid_val:
        uuid_parts.append(uuid_val)
    if advertising_id:
        uuid_parts.append(advertising_id)
    uuid_str = ";".join(uuid_parts) if uuid_parts else ""
    
    # Format cookies
    ds_user_id = session_data.get("authorization_data", {}).get("ds_user_id", "")
    sessionid = session_data.get("authorization_data", {}).get("sessionid", "")
    mid = session_data.get("mid", "")
    
    # Build cookies string with rur, mid, ds_user_id, sessionid
    cookies_parts = []
    if ds_user_id:
        # Format: rur=HIL,ds_user_id,timestamp:hash
        cookies_parts.append(f"rur=HIL,{ds_user_id},1788275080:01fea7288b0a6508624390878b012f092c7cd6837a96601ced332144ad267b12830faa9e")
    if mid:
        cookies_parts.append(f"mid={mid}")
    if ds_user_id:
        cookies_parts.append(f"ds_user_id={ds_user_id}")
    if sessionid:
        cookies_parts.append(f"sessionid={sessionid}")
    
    cookies_str = ";".join(cookies_parts) if cookies_parts else ""
    
    # Proxy format: https:ip:port:user:pass or null
    proxy_str = proxy if proxy else ""
    
    # Email credentials
    email_str = ""
    if email and email_password:
        email_str = f"{email}:{email_password}"
    
    # Build account_string - format matches user's example
    # username:password|Instagram info|User-Agent|UUIDs|cookies|proxy|email:password
    parts = [
        f"{username}:{password}",
        instagram_info,
        user_agent,
        uuid_str,  # UUIDs separated by semicolons
        cookies_str,  # Cookies separated by semicolons
        proxy_str,
        email_str
    ]
    
    return "|".join(parts)


async def get_user_info_from_session_async(cl: Client, username: str) -> Dict:
    """Get user info (followers, following, bio, etc.) using the logged-in client (async)"""
    try:
        user_info = await asyncio.to_thread(cl.user_info_by_username, username)
        return {
            "followers": user_info.follower_count if hasattr(user_info, 'follower_count') else None,
            "following": user_info.following_count if hasattr(user_info, 'following_count') else None,
            "ig_name": user_info.full_name if hasattr(user_info, 'full_name') else None,
            "bio": user_info.biography if hasattr(user_info, 'biography') else None,
            "instagram_id": str(user_info.pk) if hasattr(user_info, 'pk') else None,
        }
    except Exception as e:
        logger.debug(f"Error fetching user info for {username}: {e}")
        return {
            "followers": None,
            "following": None,
            "ig_name": None,
            "bio": None,
            "instagram_id": None,
        }


def get_user_info_from_session(cl: Client, username: str) -> Dict:
    """Get user info (followers, following, bio, etc.) using the logged-in client (sync - kept for backward compatibility)"""
    try:
        user_info = cl.user_info_by_username(username)
        return {
            "followers": user_info.follower_count if hasattr(user_info, 'follower_count') else None,
            "following": user_info.following_count if hasattr(user_info, 'following_count') else None,
            "ig_name": user_info.full_name if hasattr(user_info, 'full_name') else None,
            "bio": user_info.biography if hasattr(user_info, 'biography') else None,
            "instagram_id": str(user_info.pk) if hasattr(user_info, 'pk') else None,
        }
    except Exception as e:
        logger.debug(f"Error fetching user info for {username}: {e}")
        return {
            "followers": None,
            "following": None,
            "ig_name": None,
            "bio": None,
            "instagram_id": None,
        }


def generate_csv_from_login_results(login_results: list, output_file: str = "accounts.csv"):
    """
    Generate CSV file with account details based on login results.
    
    Args:
        login_results: List of dicts with login results containing:
            - username
            - password (IG password)
            - email
            - email_password
            - status (ok/error/cancelled)
            - message
            - session_file
            - proxy (optional)
            - verification_code (optional)
    """
    
    # CSV columns as specified
    columns = [
        "id", "name", "type", "created_at", "updated_at", "is_managed", "workflow_id",
        "password", "verification_code", "ig_username", "ig_password", "active",
        "adspower_id", "notes", "logged_in", "avatar", "tag", "session_id", "user_id",
        "qualifying_tracking", "proxy_id", "delete_request", "deleted", "account_string",
        "upload_log_id", "previous_campaign", "price_per_account", "ufac_complete",
        "adspower_configuration", "suspended_date", "deleted_date", "two_factor",
        "log", "followers", "following", "ig_name", "bio", "instagram_id"
    ]
    
    rows = []
    current_timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    
    for result in login_results:
        username = result.get("username", "")
        password = result.get("password", "")
        email = result.get("email", "")
        email_password = result.get("email_password", "")
        status = result.get("status", "error")
        message = result.get("message", "")
        session_file = result.get("session_file")
        proxy = result.get("proxy")
        verification_code = result.get("verification_code", "")
        
        # Generate random IDs
        account_id = generate_random_id(15)
        workflow_id = generate_workflow_id(15)
        
        # Determine active status and suspended date
        is_suspended = status == "suspended"
        active = "true" if status == "ok" or status == "success" else "false"
        logged_in = "true" if status == "ok" or status == "success" else "false"
        suspended_date = current_timestamp if is_suspended else None
        
        # Generate notes
        if status == "ok" or status == "success":
            notes = f"✅ Login successful on {current_timestamp} - Account credentials verified and working"
        elif is_suspended:
            # Check the specific suspended reason
            suspended_reason = result.get("suspended_reason", "")
            if suspended_reason == "Invalid password":
                notes = f"🚫 Account suspended on {current_timestamp} - Invalid password provided"
            elif suspended_reason == "No email verification code received":
                notes = f"🚫 Account suspended on {current_timestamp} - No email verification code received within 35 seconds"
            else:
                notes = f"🚫 Account suspended on {current_timestamp} - {message}"
        else:
            notes = f"❌ Login failed on {current_timestamp} - {message}"
        
        # Extract session data
        session_id = None
        user_id = None
        account_string_value = ""
        
        # Get user info from login results first (if available)
        followers = result.get("followers")
        following = result.get("following")
        ig_name = result.get("ig_name")
        bio = result.get("bio")
        instagram_id = result.get("instagram_id")
        
        if session_file and os.path.exists(session_file):
            try:
                with open(session_file, 'r') as f:
                    session_data = json.load(f)
                
                # Extract session_id and user_id
                auth_data = session_data.get("authorization_data", {})
                session_id = auth_data.get("sessionid", "")
                user_id = auth_data.get("ds_user_id", "")
                
                # Get user agent
                user_agent = session_data.get("user_agent", "")
                
                # Generate account_string
                account_string_value = generate_account_string(
                    username, password, session_data, user_agent, "", proxy, email, email_password
                )
                
                # Try to get user info if login was successful and not already in results
                if (status == "ok" or status == "success") and not followers:
                    try:
                        cl = Client()
                        cl.load_settings(session_file)
                        user_info = get_user_info_from_session(cl, username)
                        if not followers:
                            followers = user_info.get("followers")
                        if not following:
                            following = user_info.get("following")
                        if not ig_name:
                            ig_name = user_info.get("ig_name")
                        if not bio:
                            bio = user_info.get("bio")
                        if not instagram_id:
                            instagram_id = user_info.get("instagram_id")
                    except Exception as e:
                        logger.debug(f"Could not fetch user info for {username}: {e}")
                        # Continue without user info
                
            except Exception as e:
                logger.debug(f"Error reading session file {session_file}: {e}")
        
        # Parse proxy to get proxy_id (just the proxy URL)
        proxy_id = proxy if proxy else None
        
        # Build row
        row = {
            "id": account_id,
            "name": username,
            "type": "scraping",
            "created_at": current_timestamp,
            "updated_at": current_timestamp,
            "is_managed": "true",
            "workflow_id": workflow_id,
            "password": email_password if email_password else "null",
            "verification_code": verification_code if verification_code else "null",
            "ig_username": username,
            "ig_password": password,
            "active": active,
            "adspower_id": None,
            "notes": notes,
            "logged_in": logged_in,
            "avatar": None,
            "tag": "cyber scraping",
            "session_id": session_id if session_id else None,
            "user_id": user_id if user_id else None,
            "qualifying_tracking": None,
            "proxy_id": proxy_id,
            "delete_request": None,
            "deleted": None,
            "account_string": account_string_value if account_string_value else "null",
            "upload_log_id": None,
            "previous_campaign": None,
            "price_per_account": "0.0",
            "ufac_complete": "false",
            "adspower_configuration": None,
            "suspended_date": suspended_date if suspended_date else "null",
            "deleted_date": None,
            "two_factor": None,
            "log": None,
            "followers": followers if followers else None,
            "following": following if following else None,
            "ig_name": ig_name if ig_name else None,
            "bio": bio if bio else None,
            "instagram_id": instagram_id if instagram_id else None,
        }
        
        rows.append(row)
    
    # Write CSV file - use absolute path
    output_path = os.path.abspath(output_file)
    with open(output_path, 'w', newline='', encoding='utf-8') as f:
        writer = csv.DictWriter(f, fieldnames=columns)
        writer.writeheader()
        # Convert None values and empty strings to "null" string for CSV
        # Exception: price_per_account should always be "0.0" (never "null")
        csv_rows = []
        for row in rows:
            csv_row = {}
            for key, value in row.items():
                if key == "price_per_account":
                    # price_per_account should always be "0.0" if not provided
                    csv_row[key] = value if value and value != "null" else "0.0"
                elif value is None or value == "":
                    csv_row[key] = "null"  # "null" string for None or empty values
                else:
                    csv_row[key] = value
            csv_rows.append(csv_row)
        writer.writerows(csv_rows)
    
    logger.info(f"✅ Generated CSV file: {output_path} with {len(rows)} accounts")
    
    # Return both path and the CSV data for API response
    return {
        "path": output_path,
        "filename": os.path.basename(output_path),
        "data": rows,  # Return the data as list of dicts
        "count": len(rows)
    }


def save_accounts_to_supabase(accounts_data: List[Dict]) -> Dict:
    """
    Save or update accounts to Supabase.
    Uses upsert based on ig_username to avoid duplicates.
    Updates existing records if they exist.
    
    Args:
        accounts_data: List of account dictionaries (rows from CSV generation)
    
    Returns:
        Dict with success count, failed count, and any errors
    """
    if not supabase:
        logger.warning("⚠️ Supabase client not available, skipping database save")
        return {
            "success": 0,
            "failed": len(accounts_data),
            "errors": ["Supabase client not initialized"]
        }
    
    success_count = 0
    failed_count = 0
    errors = []
    
    for account in accounts_data:
        try:
            # Prepare account data for Supabase
            # Convert "null" strings back to None, but handle NOT NULL constraints
            supabase_account = {}
            
            # Define default values for fields that might have NOT NULL constraints
            defaults = {
                "deleted": False,  # Boolean field with NOT NULL constraint
                "delete_request": False,  # Boolean field likely has NOT NULL
                "price_per_account": 0.0,  # Numeric default
                "followers": 0,  # Integer field with NOT NULL constraint
                "following": 0,  # Integer field with NOT NULL constraint
            }
            
            for key, value in account.items():
                # Check for null values (both string "null" and Python None)
                if value == "null" or value == "" or value is None:
                    # Check if this field has a NOT NULL constraint and use default
                    if key in defaults:
                        supabase_account[key] = defaults[key]
                    else:
                        supabase_account[key] = None
                elif key == "price_per_account":
                    # Convert to float if it's "0.0" or a number string
                    try:
                        supabase_account[key] = float(value) if value else 0.0
                    except (ValueError, TypeError):
                        supabase_account[key] = 0.0
                elif key == "active" or key == "logged_in" or key == "is_managed" or key == "ufac_complete":
                    # Convert "true"/"false" strings to boolean
                    supabase_account[key] = value.lower() == "true" if isinstance(value, str) else bool(value)
                elif key in ["deleted", "delete_request"]:
                    # Boolean fields that might have NOT NULL constraint - always ensure boolean value
                    if value == "null" or value == "" or value is None:
                        supabase_account[key] = False  # Default to False for NOT NULL boolean fields
                    else:
                        supabase_account[key] = value.lower() == "true" if isinstance(value, str) else bool(value)
                elif key in ["followers", "following"]:
                    # Integer fields that might have NOT NULL constraints
                    if value and value != "null" and value is not None:
                        try:
                            supabase_account[key] = int(value)
                        except (ValueError, TypeError):
                            # Use default if conversion fails
                            supabase_account[key] = defaults.get(key, 0)
                    else:
                        # Use default if null/empty
                        supabase_account[key] = defaults.get(key, 0)
                elif key in ["upload_log_id", "previous_campaign"]:
                    # Try to convert numeric fields (nullable)
                    if value and value != "null":
                        try:
                            supabase_account[key] = int(value)
                        except (ValueError, TypeError):
                            supabase_account[key] = None
                    else:
                        supabase_account[key] = None
                else:
                    supabase_account[key] = value
            
            # Ensure required NOT NULL fields have values
            # Set defaults for fields that have NOT NULL constraints
            not_null_fields = {
                "deleted": False,
                "delete_request": False,
                "price_per_account": 0.0,
                "followers": 0,
                "following": 0,
            }
            
            for field, default_value in not_null_fields.items():
                if field not in supabase_account or supabase_account.get(field) is None:
                    supabase_account[field] = default_value
            
            # Handle suspended_date - convert "null" string to None, keep timestamp if present
            if supabase_account.get("suspended_date") == "null":
                supabase_account["suspended_date"] = None
            # If suspended_date has a timestamp value, keep it (it's already a string timestamp)
            
            # Handle UUID fields - user_id and possibly others might be UUID type
            # Convert Instagram user_id (numeric string) to None if it's not a valid UUID
            # The 'id' field should already be a UUID format from generate_random_id
            
            # If user_id looks like a numeric string (Instagram ID), set to None
            # as Supabase expects UUID format
            if supabase_account.get("user_id"):
                user_id_val = supabase_account.get("user_id")
                # Check if it's a numeric string (Instagram ID) vs UUID format
                if isinstance(user_id_val, str) and user_id_val.isdigit():
                    # It's an Instagram numeric ID, not a UUID - set to None
                    supabase_account["user_id"] = None
                # If it's already None or empty, keep it as None
            
            # Use upsert - update if ig_username exists, insert if not
            # Check if account exists first
            existing = supabase.table("accounts").select("ig_username").eq("ig_username", supabase_account.get("ig_username")).execute()
            
            if existing.data and len(existing.data) > 0:
                # Update existing record
                response = supabase.table("accounts").update(supabase_account).eq("ig_username", supabase_account.get("ig_username")).execute()
            else:
                # Insert new record
                response = supabase.table("accounts").insert(supabase_account).execute()
            
            if response.data:
                success_count += 1
                logger.debug(f"✅ Saved/updated account: {account.get('ig_username')}")
            else:
                failed_count += 1
                errors.append(f"Failed to save {account.get('ig_username')}: No data returned")
                
        except Exception as e:
            failed_count += 1
            error_msg = f"Error saving account {account.get('ig_username', 'unknown')}: {str(e)}"
            errors.append(error_msg)
            logger.error(error_msg)
    
    logger.info(f"📊 Supabase save complete: {success_count} succeeded, {failed_count} failed")
    
    return {
        "success": success_count,
        "failed": failed_count,
        "errors": errors[:10]  # Limit to first 10 errors to avoid huge responses
    }


def ensure_sessions_dir() -> None:
    """Ensure the sessions directory exists"""
    os.makedirs(SESSIONS_DIR, exist_ok=True)
    logger.debug(f"Sessions directory ensured: {SESSIONS_DIR}")


def save_successful_login_to_verify(username: str, password: str, email: str = "", email_password: str = "") -> None:
    """
    Save successful login to verify.csv file.
    Format: username:password:email:emailpassword
    """
    try:
        # Ensure verify.csv exists or create it
        file_exists = os.path.exists(VERIFY_CSV) and os.path.getsize(VERIFY_CSV) > 0
        
        # Check if username already exists in verify.csv
        if file_exists:
            with open(VERIFY_CSV, 'r', encoding='utf-8') as f:
                reader = csv.reader(f)
                for row in reader:
                    if row and len(row) > 0 and row[0].strip() == username:
                        logger.debug(f"Username {username} already exists in verify.csv, skipping")
                        return
        
        # Append successful login to verify.csv
        with open(VERIFY_CSV, 'a', newline='', encoding='utf-8') as f:
            writer = csv.writer(f)
            # Write header if file is new
            if not file_exists:
                writer.writerow(["username", "password", "email", "email_password"])
            # Write account data in format: username:password:email:emailpassword
            writer.writerow([username, password, email, email_password])
        
        logger.info(f"✅ Saved successful login to verify.csv: {username}")
    except Exception as e:
        logger.error(f"Failed to save successful login to verify.csv for {username}: {e}")


def session_file_for(username: str) -> str:
    safe = username.strip().lower()
    return os.path.join(SESSIONS_DIR, f"{safe}_session.json")


def load_proxies_from_file() -> List[str]:
    """
    Load proxies from proxies.txt file.
    Each line should contain one proxy in format: scheme://user:pass@host:port
    Returns empty list if file doesn't exist or is empty.
    """
    global _proxies_cache
    
    # Return cached proxies if available
    if _proxies_cache is not None:
        return _proxies_cache
    
    proxies = []
    if os.path.exists(PROXIES_TXT):
        try:
            with open(PROXIES_TXT, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    # Skip empty lines and comments
                    if line and not line.startswith('#'):
                        proxies.append(line)
            logger.info(f"✅ Loaded {len(proxies)} proxies from {PROXIES_TXT}")
        except Exception as e:
            logger.error(f"Error loading proxies from {PROXIES_TXT}: {e}")
    else:
        logger.debug(f"Proxies file not found: {PROXIES_TXT}")
    
    # Cache the proxies
    _proxies_cache = proxies
    return proxies


def get_random_proxy() -> Optional[str]:
    """
    Get a random proxy from proxies.txt file.
    Returns None if no proxies are available.
    Ensures proxy has proper scheme (http:// or https://).
    """
    proxies = load_proxies_from_file()
    if proxies:
        proxy = random.choice(proxies)
        # Ensure proxy has proper scheme (instagrapi requires it)
        if proxy and not proxy.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
            # Add http:// if no scheme
            proxy = f"http://{proxy}"
        logger.debug(f"Randomly selected proxy: {mask_proxy(proxy)}")
        return proxy
    return None


def reload_proxies_cache() -> None:
    """
    Reload proxies from proxies.txt file (clears cache).
    Useful when proxies.txt is updated during runtime.
    """
    global _proxies_cache
    _proxies_cache = None
    logger.info("Proxy cache cleared - proxies will be reloaded on next request")


def mask_proxy(proxy_url: str) -> str:
    try:
        if "://" in proxy_url and "@" in proxy_url:
            scheme, rest = proxy_url.split("://", 1)
            creds, host = rest.split("@", 1)
            return f"{scheme}://***@{host}"
        return proxy_url
    except Exception:
        return "***"


def extract_email_parts(email_address: str) -> Tuple[str, str]:
    """Extract username and domain from email address"""
    if "@" in email_address:
        username, domain = email_address.split("@", 1)
        return username, domain
    return email_address, ""


def get_code_from_email_for_challenge(username: str, email_address: str, imap_password: str = DEFAULT_IMAP_PASSWORD, max_wait: int = 35, stop_event: Optional[threading.Event] = None) -> Optional[str]:
    """
    Get Instagram challenge code from email using IMAP.
    Based on instagrapi documentation pattern - checks UNSEEN emails and matches Instagram's email format.
    """
    email_username, domain = extract_email_parts(email_address)
    
    if not domain:
        logger.error(f"Invalid email address: {email_address}")
        return False
    
    # Use only the domain (no imap.domain.com fallback)
    imap_server = domain
    
    try:
        logger.info(f"Attempting IMAP connection to {imap_server}:{DEFAULT_IMAP_PORT} for {username}")
        
        # Connect to IMAP server
        mail = imaplib.IMAP4_SSL(imap_server, DEFAULT_IMAP_PORT)
        
        # Login
        mail.login(email_address, imap_password)
        logger.info(f"Successfully connected to IMAP server {imap_server}")
        
        # Select inbox
        mail.select("inbox")
        logger.info(f"🔔 Challenge requested, checking UNSEEN emails for Instagram verification code...")
        
        start_time = time.time()
        code = None
        
        # Poll for new emails containing the code - following instagrapi pattern
        while time.time() - start_time < max_wait and code is None:
            # Check for stop event
            if stop_event and stop_event.is_set():
                logger.info(f"Stop event triggered for {username}, stopping email check...")
                break
            
            try:
                # Check UNSEEN emails first (like instagrapi example)
                result, data = mail.search(None, "(UNSEEN)")
                if result != "OK":
                    logger.debug(f"Error searching UNSEEN emails: {result}")
                    time.sleep(3)
                    continue
                
                if not data or not data[0]:
                    # No unseen emails, wait and check again
                    elapsed = time.time() - start_time
                    logger.info(f"⏳ No unseen emails yet, waiting... ({elapsed:.1f}s elapsed)")
                    time.sleep(3)
                    continue
                
                # Get email IDs
                email_str = data[0].decode() if isinstance(data[0], bytes) else data[0]
                if not email_str.strip():
                    elapsed = time.time() - start_time
                    logger.info(f"⏳ No unseen emails yet, waiting... ({elapsed:.1f}s elapsed)")
                    time.sleep(3)
                    continue
                
                ids = email_str.split()
                logger.info(f"📧 Found {len(ids)} unseen email(s), checking for Instagram code...")
                
                # Check emails in reverse order (newest first) - like instagrapi example
                for num in reversed(ids):
                    try:
                        # Mark as read
                        mail.store(num, "+FLAGS", "\\Seen")
                        
                        # Fetch email
                        result, data = mail.fetch(num, "(RFC822)")
                        if result != "OK":
                            continue
                        
                        # Parse email - data[0][1] is always bytes for RFC822
                        email_body = data[0][1]
                        msg = email.message_from_bytes(email_body)
                        
                        payloads = msg.get_payload()
                        if not isinstance(payloads, list):
                            payloads = [msg]
                        
                        for payload in payloads:
                            body = payload.get_payload(decode=True).decode(errors='ignore')
                            if not body:
                                continue
                            
                            # Check if it's HTML email (like instagrapi example)
                            if "<div" not in body:
                                continue
                            
                            # Check if username is in email body (like instagrapi example)
                            username_pattern = r">([^>]*?({u})[^<]*?)<".format(u=username)
                            match = re.search(username_pattern, body, re.IGNORECASE)
                            if not match:
                                continue
                            
                            logger.debug(f"Match from email for {username}: {match.group(1)}")
                            
                            # Look for 6-digit code using Instagram's format: >(\d{6})<
                            match = re.search(r">(\d{6})<", body)
                            if not match:
                                logger.debug('Skip this email, "code" not found in format >XXXXXX<')
                                continue
                            
                            potential_code = match.group(1)
                            if len(potential_code) == 6 and potential_code.isdigit():
                                code = potential_code
                                logger.info(f"✅ Found verification code {code} in email from {imap_server} for {username}")
                                mail.close()
                                mail.logout()
                                return code
                    
                    except Exception as e:
                        logger.debug(f"Error processing email {num}: {e}")
                        continue
                
                # If no code found, wait and check again
                if not code:
                    time.sleep(3)
            
            except Exception as e:
                logger.warning(f"Error searching emails: {e}")
                time.sleep(3)
        
        mail.close()
        mail.logout()
        
        if not code:
            elapsed = time.time() - start_time
            if elapsed >= 35:
                logger.warning(f"⚠️ No verification code found in emails from {imap_server} after 35 seconds")
                logger.warning(f"📧 Instagram did not send email code within timeout period")
                logger.warning(f"🛑 Stopping email check and marking account as suspended")
            else:
                logger.warning(f"⚠️ No verification code found in emails from {imap_server} after {max_wait} seconds")
                logger.warning(f"📧 Instagram may not have sent an email code, or it may require SMS verification instead")
                logger.warning(f"💡 Possible reasons: Instagram requires SMS, code was already used, or account security settings")
        
        return code if code else False
            
    except imaplib.IMAP4.error as e:
        logger.error(f"IMAP error connecting to {imap_server}:{DEFAULT_IMAP_PORT}: {e}")
        return False
    except Exception as e:
        logger.error(f"Error connecting to IMAP server {imap_server}: {e}")
        return False


def get_verification_code_from_email(email_address: str, imap_password: str = DEFAULT_IMAP_PASSWORD, max_wait: int = 120) -> Optional[str]:
    """
    Get Instagram verification code from email using IMAP.
    Uses only the domain (no imap.domain.com fallback)
    """
    username, domain = extract_email_parts(email_address)
    
    if not domain:
        logger.error(f"Invalid email address: {email_address}")
        return None
    
    # Use only the domain (no imap.domain.com fallback)
    imap_server = domain
    
    try:
        logger.info(f"Attempting IMAP connection to {imap_server}:{DEFAULT_IMAP_PORT} for {email_address}")
        
        # Connect to IMAP server
        mail = imaplib.IMAP4_SSL(imap_server, DEFAULT_IMAP_PORT)
        
        # Login
        mail.login(email_address, imap_password)
        logger.info(f"Successfully connected to IMAP server {imap_server}")
        
        # Select inbox
        mail.select('inbox')
        
        # Search for Instagram verification emails from the last 5 minutes
        # Look for emails from Instagram or containing verification code
        search_criteria = '(FROM "instagram.com" OR FROM "mail.instagram.com" OR SUBJECT "code" OR SUBJECT "verification")'
        
        start_time = time.time()
        code = None
        
        while time.time() - start_time < max_wait and code is None:
            try:
                # Search for recent emails
                status, messages = mail.search(None, search_criteria)
                
                if status == 'OK':
                    email_ids = messages[0].split()
                    
                    # Check most recent emails first
                    for email_id in reversed(email_ids[-10:]):  # Check last 10 emails
                        try:
                            status, msg_data = mail.fetch(email_id, '(RFC822)')
                            if status == 'OK':
                                email_body = msg_data[0][1]
                                email_message = email.message_from_bytes(email_body)
                                
                                # Get email content
                                body = ""
                                if email_message.is_multipart():
                                    for part in email_message.walk():
                                        if part.get_content_type() == "text/plain":
                                            body = part.get_payload(decode=True).decode('utf-8', errors='ignore')
                                            break
                                else:
                                    body = email_message.get_payload(decode=True).decode('utf-8', errors='ignore')
                                
                                # Search for 6-digit verification code
                                # Instagram codes are usually 6 digits
                                code_patterns = [
                                    r'\b(\d{6})\b',  # Any 6-digit code
                                    r'code[:\s]*(\d{6})',  # Code followed by 6 digits
                                    r'verification[:\s]*code[:\s]*(\d{6})',  # Verification code
                                    r'(\d{6})[^\d]',  # 6 digits followed by non-digit
                                ]
                                
                                for pattern in code_patterns:
                                    matches = re.findall(pattern, body, re.IGNORECASE)
                                    if matches:
                                        code = matches[0]
                                        logger.info(f"Found verification code {code} in email from {imap_server}")
                                        break
                                
                                if code:
                                    break
                        except Exception as e:
                            logger.warning(f"Error processing email {email_id}: {e}")
                            continue
                
                if not code:
                    time.sleep(5)  # Wait 5 seconds before checking again
                    
            except Exception as e:
                logger.warning(f"Error searching emails: {e}")
                time.sleep(5)
        
        mail.close()
        mail.logout()
        
        if code:
            return code
        else:
            logger.warning(f"No verification code found in emails from {imap_server}")
            return None
            
    except imaplib.IMAP4.error as e:
        logger.error(f"IMAP error connecting to {imap_server}:{DEFAULT_IMAP_PORT}: {e}")
        return None
    except Exception as e:
        logger.error(f"Error connecting to IMAP server {imap_server}: {e}")
        return None


def generate_random_user_agent(locale: str = DEFAULT_LOCALE) -> str:
    android_versions = [
        (26, "8.0.0"),
        (27, "8.1.0"),
        (28, "9"),
        (29, "10"),
        (30, "11"),
    ]
    dpis = [320, 400, 420, 440, 480]
    resolutions = [(1080, 1920), (1440, 2560), (1080, 2340), (1080, 2400)]
    brands_models = [
        ("Xiaomi", "MI 5s", "capricorn"),
        ("Samsung", "SM-G960F", "starlte"),
        ("Google", "Pixel 3", "blueline"),
        ("OnePlus", "ONEPLUS A6013", "fajita"),
        ("Huawei", "P20", "emily"),
    ]
    version_code = random.choice([194, 239, 301, 318])
    version_name = random.choice(["194.0.0.36.172", "239.0.0.14.111", "301.0.0.0.89", "318.0.0.30.109"])

    (sdk, release) = random.choice(android_versions)
    dpi = random.choice(dpis)
    width, height = random.choice(resolutions)
    brand, model, device = random.choice(brands_models)

    return (
        f"Instagram {version_name} Android ({sdk}/{release}; {dpi}dpi; {width}x{height}; "
        f"{brand}; {model}; {device}; qcom; {locale}; {version_code})"
    )


def parse_txt(txt_path: str) -> List[Dict[str, str]]:
    """
    Parse TXT file with format: username:password:email:emailpassword
    Example: stephenburrell848236:Test$20231704:max9951980523@scalingtothemoon.com:Heritech@098A!2
    
    Format: username:password:email:emailpassword
    - All fields separated by colons
    - Format: username:password:email:emailpassword
    - If emailpassword contains colons, join remaining parts
    - One account per line
    """
    logger.info(f"Reading TXT: {os.path.abspath(txt_path)}")
    accounts: List[Dict[str, str]] = []
    count = 0
    
    # Handle both file paths and raw text content
    if isinstance(txt_path, str) and os.path.exists(txt_path):
        with open(txt_path, "r", encoding="utf-8") as f:
            lines = f.readlines()
    else:
        # Treat as raw text content (from API endpoint)
        content = txt_path if isinstance(txt_path, str) else ""
        lines = content.splitlines() if content else []
    
    for line_num, line in enumerate(lines, 1):
        # Strip whitespace from beginning and end, but preserve internal spaces
        line = line.strip()
        
        # Skip empty lines, comments, lines that are just commas, or whitespace-only
        if not line or line.startswith("#") or line in [",", ",", ""]:
            continue
        
        # Remove any trailing commas or special characters that might interfere
        line = line.rstrip(",\n\r")
        
        # Split by colon
        parts = [part.strip() for part in line.split(":")]
        
        # Expected format: username:password:email:emailpassword (4 parts)
        # Minimum required: username:password (2 parts)
        if len(parts) >= 2:
            username = parts[0].strip()
            password = parts[1].strip()
            
            # Extract email (part 2) if available
            email = parts[2].strip() if len(parts) > 2 and parts[2] else ""
            
            # Extract emailpassword (part 3 and beyond)
            # Join all parts from index 3 onwards to handle email passwords with colons
            if len(parts) > 3:
                emailpassword = ":".join(parts[3:]).strip()
            elif len(parts) == 3:
                # Only 3 parts: username:password:email (no emailpassword provided)
                emailpassword = DEFAULT_IMAP_PASSWORD
            else:
                # Only 2 parts: username:password (no email, no emailpassword)
                emailpassword = DEFAULT_IMAP_PASSWORD
            
            # Validate required fields
            if not username or not password:
                logger.warning(f"Skipping line {line_num}: missing username or password (username='{username}', password='{password[:5]}...')")
                continue
            
            accounts.append(
                {
                    "username": username,
                    "password": password,
                    "email": email,
                    "email_password": emailpassword,  # Store as email_password internally
                    "totp_secret": "",
                    "proxy": "",
                    "locale": DEFAULT_LOCALE,
                    "timezone_offset": DEFAULT_TZ_OFFSET,
                }
            )
            count += 1
            logger.debug(f"Parsed line {line_num}: username={username}, email={email if email else 'default'}")
        else:
            logger.warning(f"Skipping line {line_num}: invalid format (expected username:password:email:emailpassword, got {len(parts)} parts)")
    
    logger.info(f"Loaded {count} accounts from TXT")
    return accounts


def parse_csv(csv_path: str) -> List[Dict[str, str]]:
    logger.info(f"Reading CSV: {os.path.abspath(csv_path)}")
    with open(csv_path, newline="", encoding="utf-8") as f:
        reader = csv.DictReader(f)
        accounts: List[Dict[str, str]] = []
        count = 0
        for row in reader:
            # Normalize keys
            normalized = {k.strip().lower(): (v or "").strip() for k, v in row.items()}
            # Expected fields
            username = normalized.get("username")
            password = normalized.get("password")
            totp_secret = (
                normalized.get("authfa")
                or normalized.get("2fa")
                or normalized.get("totp")
                or normalized.get("totp_secret")
            )
            proxy = normalized.get("proxy")
            locale = normalized.get("locale") or DEFAULT_LOCALE
            tz_offset_str = normalized.get("timezone_offset")
            if tz_offset_str:
                try:
                    tz_offset = int(tz_offset_str)
                except ValueError:
                    tz_offset = DEFAULT_TZ_OFFSET
            else:
                tz_offset = DEFAULT_TZ_OFFSET

            if not username or not password:
                continue

            accounts.append(
                {
                    "username": username,
                    "password": password,
                    "totp_secret": totp_secret or "",
                    "proxy": proxy or "",
                    "locale": locale,
                    "timezone_offset": tz_offset,
                }
            )
            count += 1
    logger.info(f"Loaded {count} accounts from CSV")
    return accounts


def parse_accounts_file(file_path: str) -> List[Dict[str, str]]:
    """
    Parse accounts file (CSV or TXT format).
    TXT format: username:password:email:emailpassword
    CSV format: standard CSV with columns
    """
    if not os.path.exists(file_path):
        raise FileNotFoundError(f"File not found: {file_path}")
    
    # Detect file type by extension
    file_ext = os.path.splitext(file_path)[1].lower()
    
    if file_ext == ".txt":
        return parse_txt(file_path)
    else:
        # Default to CSV
        return parse_csv(file_path)


def login_account(account: Dict[str, str], stop_event: Optional[threading.Event] = None) -> Dict[str, Optional[str]]:
    if stop_event and stop_event.is_set():
        return {"username": account.get("username", ""), "status": "cancelled", "message": "stopped", "session_file": None}

    username = account["username"]
    logger.info(f"Processing: {username}")
    session_path = session_file_for(username)

    if os.path.exists(session_path) and os.path.getsize(session_path) > 0:
        logger.info(f"Skip existing session for {username}")
        return {
            "username": username,
            "status": "skipped",
            "message": "session exists",
            "session_file": session_path,
        }

    cl = Client()
    try:
        # Reduce potential long waits
        try:
            cl.request_timeout = 15  # seconds
        except Exception:
            pass

        # Proxy requirement - check account proxy, then proxies.txt (random, only source)
        proxy_to_use = None
        proxy_source = None
        account_proxy = account.get("proxy")
        if account_proxy:
            proxy_to_use = account_proxy
            proxy_source = "account"
        else:
            # Try to get random proxy from proxies.txt (only source)
            random_proxy = get_random_proxy()
            if random_proxy:
                proxy_to_use = random_proxy
                proxy_source = "proxies.txt (random)"
            else:
                # No proxy available from proxies.txt
                proxy_to_use = None
                proxy_source = "none (proxies.txt empty or not found)"
        
        if proxy_to_use:
            try:
                # Ensure proxy has proper scheme (instagrapi requires it)
                if not proxy_to_use.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
                    proxy_to_use = f"http://{proxy_to_use}"
                
                logger.info(f"🔥 PROXY DETECTED - USING {proxy_source.upper()} PROXY FOR {username.upper()}: {mask_proxy(proxy_to_use)} 🔥")
                cl.set_proxy(proxy_to_use)
            except Exception as e:
                logger.warning(f"Failed to set proxy for {username}: {e}")
                logger.warning(f"Proxy format was: {mask_proxy(proxy_to_use)}")
        else:
            logger.warning("⚠  NO PROXY FOUND! ⚠")
            logger.warning(f"❌ {username} will connect without proxy ❌")
        
        # Set up challenge_code_handler for automatic challenge handling
        email_address = username if "@" in username else account.get("email") or account.get("email_address")
        email_password = account.get("email_password") or DEFAULT_IMAP_PASSWORD
        
        if email_address:
            def challenge_code_handler(insta_username, choice):
                """Handle Instagram challenge codes via IMAP"""
                if choice == ChallengeChoice.EMAIL:
                    logger.info(f"Challenge via EMAIL requested for {insta_username}")
                    logger.info(f"📧 Checking email {email_address} for verification code...")
                    code = get_code_from_email_for_challenge(insta_username, email_address, email_password, stop_event=stop_event)
                    if code:
                        logger.info(f"✅ Challenge code retrieved for {insta_username}: {code}")
                        return code
                    else:
                        # No code received after 35 seconds - mark as suspended
                        logger.warning(f"❌ Failed to retrieve challenge code for {insta_username} after 35 seconds")
                        logger.warning(f"🚫 Marking account {insta_username} as SUSPENDED")
                        logger.warning(f"💡 Instagram did not send email verification code within timeout")
                        # Return special value to indicate suspension
                        return "SUSPENDED_NO_CODE"
                elif choice == ChallengeChoice.SMS:
                    logger.warning(f"⚠️ Challenge via SMS requested for {insta_username} (SMS verification not supported)")
                    logger.warning(f"💡 Account requires SMS verification - manual intervention needed")
                    return False
                else:
                    logger.warning(f"⚠️ Unknown challenge choice: {choice}")
                    return False
            
            cl.challenge_code_handler = challenge_code_handler
            logger.info(f"✅ Challenge code handler set for {username} using email {email_address}")

        # Locale and timezone
        cl.set_locale(account.get("locale") or DEFAULT_LOCALE)
        tz_offset = int(account.get("timezone_offset") or DEFAULT_TZ_OFFSET)
        cl.set_timezone_offset(tz_offset)

        # Random UA
        user_agent = generate_random_user_agent(account.get("locale") or DEFAULT_LOCALE)
        try:
            cl.set_user_agent(user_agent)
            logger.info(f"User-Agent set for {username}")
        except Exception:
            logger.info(f"User-Agent set via settings for {username}")

        if stop_event and stop_event.is_set():
            return {"username": username, "status": "cancelled", "message": "stopped", "session_file": None}

        # Optional small randomized delay to reduce flapping
        time.sleep(random.uniform(0.2, 0.8))

        username = account["username"]
        password = account["password"]

        if stop_event and stop_event.is_set():
            return {"username": username, "status": "cancelled", "message": "stopped", "session_file": None}

        # Attempt login - challenge_code_handler will automatically handle challenges
        try:
            cl.login(username, password)
        except Exception as login_exception:
            # Check if login failed due to suspended/no code or invalid password
            error_msg = str(login_exception).lower()
            
            # Check for invalid password
            if "bad_password" in error_msg or "invalid password" in error_msg or "wrong password" in error_msg:
                logger.warning(f"🚫 Invalid password for {username}")
                return {
                    "username": username,
                    "status": "suspended",
                    "message": "Invalid password - login failed",
                    "session_file": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                    "suspended_reason": "Invalid password",
                }
            elif "challenge" in error_msg or "verification" in error_msg:
                # Check if challenge handler returned suspended status
                logger.warning(f"🚫 Login failed for {username} - possible suspended/no code response")
                # Return suspended status
                return {
                    "username": username,
                    "status": "suspended",
                    "message": "Login failed - no verification code received within 35 seconds",
                    "session_file": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                    "suspended_reason": "No email verification code received",
                }
            else:
                # Other login error - re-raise
                raise

        # Save session settings
        ensure_sessions_dir()
        cl.dump_settings(session_path)
        
        # Save successful login to verify.csv
        save_successful_login_to_verify(username, password, email_address, email_password)

        # Pull settings to expose useful info
        settings = cl.get_settings()
        ua = None
        try:
            ua = settings.get("user_agent") if isinstance(settings, dict) else None
        except Exception:
            ua = None

        logger.info(f"Login success for {username}; session saved")
        
        # Try to get user info (using sync version since this function is sync)
        followers = None
        following = None
        ig_name = None
        bio = None
        instagram_id = None
        try:
            user_info = cl.user_info_by_username(username)
            followers = user_info.follower_count if hasattr(user_info, 'follower_count') else None
            following = user_info.following_count if hasattr(user_info, 'following_count') else None
            ig_name = user_info.full_name if hasattr(user_info, 'full_name') else None
            bio = user_info.biography if hasattr(user_info, 'biography') else None
            instagram_id = str(user_info.pk) if hasattr(user_info, 'pk') else None
        except Exception as e:
            logger.debug(f"Could not fetch user info for {username}: {e}")
        
        return {
            "username": username,
            "status": "ok",
            "message": "logged in and session saved",
            "session_file": session_path,
            "user_agent": ua or user_agent,
            "proxy": proxy_to_use,
            "email": email_address,
            "email_password": email_password,
            "followers": followers,
            "following": following,
            "ig_name": ig_name,
            "bio": bio,
            "instagram_id": instagram_id,
        }
    except Exception as e:
        logger.error(f"Login error for {username}: {e}")
        # Extract email info even on error
        email_address = username if "@" in username else account.get("email") or account.get("email_address")
        email_password = account.get("email_password") or DEFAULT_IMAP_PASSWORD
        
        # Check if it's an invalid password error
        error_msg = str(e).lower()
        if "bad_password" in error_msg or "invalid password" in error_msg or "wrong password" in error_msg:
            logger.warning(f"🚫 Invalid password for {username} - marking as suspended")
            return {
                "username": username,
                "status": "suspended",
                "message": f"Invalid password - {str(e)}",
                "session_file": None,
                "email": email_address,
                "email_password": email_password,
                "password": account.get("password", ""),
                "suspended_reason": "Invalid password",
            }
        
        return {
            "username": username,
            "status": "error",
            "message": str(e),
            "session_file": None,
            "email": email_address,
            "email_password": email_password,
            "password": account.get("password", ""),
        }


def write_noactive_csv(failed_accounts: List[Dict[str, str]]) -> None:
    if not failed_accounts:
        return
    # Always write 3 columns as per provided CSV: username,password,authfa
    file_exists = os.path.exists(NOACTIVE_CSV) and os.path.getsize(NOACTIVE_CSV) > 0
    with open(NOACTIVE_CSV, "a", newline="", encoding="utf-8") as f:
        writer = csv.DictWriter(f, fieldnames=["username", "password", "authfa"])
        if not file_exists:
            writer.writeheader()
        for acc in failed_accounts:
            writer.writerow(
                {
                    "username": acc.get("username", ""),
                    "password": acc.get("password", ""),
                    "authfa": acc.get("totp_secret", ""),
                }
            )
    logger.info(f"Appended {len(failed_accounts)} to '{os.path.basename(NOACTIVE_CSV)}'")


def remove_failed_from_verify(csv_path: str, failed_usernames: Set[str]) -> None:
    if not failed_usernames:
        return
    temp_path = f"{csv_path}.tmp"
    kept = 0
    removed = 0
    with open(csv_path, newline="", encoding="utf-8") as src, open(temp_path, "w", newline="", encoding="utf-8") as dst:
        reader = csv.DictReader(src)
        fieldnames = reader.fieldnames or ["username", "password", "authfa"]
        writer = csv.DictWriter(dst, fieldnames=fieldnames)
        writer.writeheader()
        for row in reader:
            uname = (row.get("username") or "").strip()
            if uname and uname in failed_usernames:
                removed += 1
                continue
            writer.writerow(row)
            kept += 1
    os.replace(temp_path, csv_path)
    logger.info(f"Removed {removed} failed accounts from verify.csv; kept {kept}")


async def batch_login_from_csv(csv_path: str, stop_event: Optional[threading.Event] = None) -> Dict[str, object]:
    """Process CSV or TXT file and login to Instagram accounts using isolated clients"""
    try:
        # Generate unique request ID for this batch operation
        request_id = str(uuid.uuid4())
        
        # Parse accounts file (supports both CSV and TXT formats)
        try:
            accounts = parse_accounts_file(csv_path)
        except KeyboardInterrupt:
            logger.warning("⚠️  KeyboardInterrupt during parsing. Stopping...")
            raise
        if not accounts:
            return {
                "status": "error",
                "message": "No valid accounts found in file",
                "summary": {"total": 0, "success": 0, "failed": 0}
            }

        logger.info(f"Processing {len(accounts)} accounts from {csv_path}")
        
        # Use asyncio for parallel processing
        num_threads = min(len(accounts), MAX_WORKERS)
        logger.info(f"Processing {len(accounts)} accounts using {num_threads} parallel tasks")

        # Check for stop event before processing
        if stop_event and stop_event.is_set():
            logger.info("Stop event triggered, cancelling batch login")
            return {
                "status": "cancelled",
                "message": "Batch login was cancelled",
                "summary": {"total": len(accounts), "success": 0, "failed": 0}
            }
        
        # Create semaphore to limit concurrent logins
        semaphore = asyncio.Semaphore(num_threads)
        
        async def run_login_with_semaphore(account):
            async with semaphore:
                # Check for stop event during processing
                if stop_event and stop_event.is_set():
                    logger.info("Stop event triggered during processing, cancelling task")
                    return {
                        "username": account.get('username', 'unknown'),
                        "status": "cancelled",
                        "message": "Task cancelled due to stop event"
                    }
                
                try:
                    result = await _process_single_login(account, request_id, stop_event)
                    logger.info(f"Completed login for {result['username']} with status: {result['status']}")
                    return result
                except Exception as e:
                    logger.error(f"Login task failed for {account.get('username', 'unknown')}: {e}")
                    return {
                        "username": account.get('username', 'unknown'),
                        "status": "error",
                        "message": f"Task execution failed: {str(e)}"
                    }
        
        # Run all login tasks concurrently
        tasks = [asyncio.create_task(run_login_with_semaphore(account)) for account in accounts]
        try:
            results = await asyncio.gather(*tasks, return_exceptions=True)
        except KeyboardInterrupt:
            logger.warning("⚠️  KeyboardInterrupt received. Cancelling all tasks...")
            # Set stop event if provided
            if stop_event:
                stop_event.set()
            # Cancel all tasks
            for task in tasks:
                if not task.done():
                    task.cancel()
            # Wait for cancellation
            try:
                await asyncio.gather(*tasks, return_exceptions=True)
            except Exception:
                pass
            raise
        
        # Handle any exceptions that occurred
        processed_results = []
        for i, result in enumerate(results):
            if isinstance(result, Exception):
                account = accounts[i]
                logger.error(f"Login task failed for {account.get('username', 'unknown')}: {result}")
                processed_results.append({
                    "username": account.get('username', 'unknown'),
                    "status": "error",
                    "message": f"Task execution failed: {str(result)}"
                })
            else:
                processed_results.append(result)
        
        results = processed_results

        # Calculate summary
        summary = {
            "total": len(accounts),
            "success": len([r for r in results if r.get("status") in ["success", "ok"]]),
            "failed": len([r for r in results if r.get("status") == "error"]),
            "suspended": len([r for r in results if r.get("status") == "suspended"]),
            "threads_used": num_threads,
            "parallel_processing": True
        }

        logger.info(f"Completed batch login. Success: {summary['success']}, Failed: {summary['failed']}, Threads: {num_threads}")
        
        # Generate CSV file with account details
        csv_filename = None
        csv_path = None
        try:
            # Prepare login results for CSV generation
            csv_results = []
            for result in results:
                # Find matching account to get email and password info
                matching_account = None
                for acc in accounts:
                    if acc.get('username') == result.get('username'):
                        matching_account = acc
                        break
                
                csv_result = {
                    "username": result.get("username", ""),
                    "password": matching_account.get("password", "") if matching_account else "",
                    "email": result.get("email") or (matching_account.get("email", "") if matching_account else ""),
                    "email_password": result.get("email_password") or (matching_account.get("email_password", "") if matching_account else ""),
                    "status": result.get("status", "error"),
                    "message": result.get("message", ""),
                    "session_file": result.get("session_file") or result.get("session_path"),
                    "proxy": result.get("proxy", ""),
                    "verification_code": result.get("verification_code", ""),
                    "followers": result.get("followers"),
                    "following": result.get("following"),
                    "ig_name": result.get("ig_name"),
                    "bio": result.get("bio"),
                    "instagram_id": result.get("instagram_id"),
                }
                csv_results.append(csv_result)
            
            # Generate CSV file
            csv_filename = f"accounts_{datetime.now().strftime('%Y%m%d_%H%M%S')}.csv"
            csv_result = generate_csv_from_login_results(csv_results, csv_filename)
            logger.info(f"✅ Generated accounts CSV file: {csv_result.get('filename', csv_filename)}")
            
            # Save to Supabase
            supabase_result = None
            if csv_result and csv_result.get("data"):
                try:
                    supabase_result = save_accounts_to_supabase(csv_result.get("data", []))
                    logger.info(f"✅ Saved {supabase_result.get('success', 0)} accounts to Supabase")
                except Exception as e:
                    logger.error(f"❌ Failed to save accounts to Supabase: {e}")
                    supabase_result = {
                        "success": 0,
                        "failed": len(csv_result.get("data", [])),
                        "errors": [str(e)]
                    }
        except Exception as e:
            logger.warning(f"⚠️ Failed to generate CSV file: {e}")
            csv_result = None
            supabase_result = None
        
        response = {
            "status": "ok", 
            "summary": summary, 
            "results": results
        }
        
        # Add CSV details to response
        if csv_result:
            response["csv"] = {
                "filename": csv_result.get("filename"),
                "path": csv_result.get("path"),
                "count": csv_result.get("count", 0),
                "data": csv_result.get("data", [])
            }
        else:
            response["csv"] = None
        
        # Add Supabase save result to response
        if supabase_result:
            response["supabase"] = supabase_result
        else:
            response["supabase"] = None
        
        return response
        
    except KeyboardInterrupt:
        logger.warning("⚠️  Batch login interrupted by user (Ctrl+C)")
        return {
            "status": "cancelled",
            "message": "Login operation cancelled by user",
            "summary": {"total": len(accounts) if 'accounts' in locals() else 0, "success": 0, "failed": 0},
            "results": []
        }
    except Exception as critical_error:
        logger.critical(f"CRITICAL ERROR in batch_login_from_csv: {critical_error}")
        import traceback
        logger.critical(f"Traceback: {traceback.format_exc()}")
        
        return {
            "status": "error", 
            "message": f"Critical error occurred: {str(critical_error)}", 
            "summary": {"total": len(accounts) if 'accounts' in locals() else 0, "success": 0, "failed": len(accounts) if 'accounts' in locals() else 0},
            "results": []
        }


async def _process_single_login(account: Dict[str, str], request_id: str, stop_event: Optional[threading.Event] = None) -> Dict[str, str]:
    """Process login for a single account using isolated client - only username and password, uses IMAP for email verification (async)"""
    username = account.get('username', '').strip()
    password = account.get('password', '').strip()
    proxy = account.get('proxy', '').strip()
    
    if not username or not password:
        return {
            "username": username,
            "status": "error",
            "message": "Missing username or password"
        }
    
    # Debug logging
    logger.info(f"Processing {username} (username and password only, IMAP for email verification)")
    logger.info(f"Account keys: {list(account.keys())}")
    
    try:
        logger.info(f"Processing login for {username}")
        
        # Create isolated client for this login
        cl = Client()
        
        # Configure client settings (synchronous setup, but fast)
        cl.set_settings({
            "app_version": "269.0.0.18.75",
            "android_version": 26,
            "android_release": "8.0.0",
            "dpi": "480dpi",
            "resolution": "1080x1920",
            "manufacturer": "Xiaomi",
            "device": "capricorn",
            "model": "MI 5s",
            "cpu": "hexa-core",
            "version_code": "301"
        })
        
        # Set user agent
        cl.set_user_agent(generate_random_user_agent())
        
        # Configure proxy - check account proxy first, then proxies.txt (random, only source)
        proxy_to_use = None
        if proxy:
            proxy_to_use = proxy
            proxy_source = "account"
        else:
            # Try to get random proxy from proxies.txt (only source)
            random_proxy = get_random_proxy()
            if random_proxy:
                proxy_to_use = random_proxy
                proxy_source = "proxies.txt (random)"
            else:
                # No proxy available from proxies.txt
                proxy_to_use = None
                proxy_source = "none (proxies.txt empty or not found)"
        
        if proxy_to_use:
            try:
                # Ensure proxy has proper scheme (instagrapi requires it)
                if not proxy_to_use.startswith(('http://', 'https://', 'socks4://', 'socks5://')):
                    proxy_to_use = f"http://{proxy_to_use}"
                
                logger.info(f"🔥 PROXY DETECTED - USING {proxy_source.upper()} PROXY FOR {username.upper()}: {mask_proxy(proxy_to_use)} 🔥")
                cl.set_proxy(proxy_to_use)
            except Exception as e:
                logger.warning(f"Failed to set proxy for {username}: {e}")
                logger.warning(f"Proxy format was: {mask_proxy(proxy_to_use)}")
        else:
            logger.warning("⚠  NO PROXY FOUND! ⚠")
            logger.warning(f"❌ {username} will connect without proxy ❌")
        
        # Set up challenge_code_handler for automatic challenge handling
        email_address = username if "@" in username else account.get("email") or account.get("email_address")
        email_password = account.get("email_password") or DEFAULT_IMAP_PASSWORD
        
        if email_address:
            def challenge_code_handler(insta_username, choice):
                """Handle Instagram challenge codes via IMAP"""
                if choice == ChallengeChoice.EMAIL:
                    logger.info(f"Challenge via EMAIL requested for {insta_username}")
                    logger.info(f"📧 Checking email {email_address} for verification code...")
                    code = get_code_from_email_for_challenge(insta_username, email_address, email_password)
                    if code:
                        logger.info(f"✅ Challenge code retrieved for {insta_username}: {code}")
                        return code
                    else:
                        # No code received after 35 seconds - mark as suspended
                        logger.warning(f"❌ Failed to retrieve challenge code for {insta_username} after 35 seconds")
                        logger.warning(f"🚫 Marking account {insta_username} as SUSPENDED")
                        logger.warning(f"💡 Instagram did not send email verification code within timeout")
                        # Return special value to indicate suspension
                        return "SUSPENDED_NO_CODE"
                elif choice == ChallengeChoice.SMS:
                    logger.warning(f"⚠️ Challenge via SMS requested for {insta_username} (SMS verification not supported)")
                    logger.warning(f"💡 Account requires SMS verification - manual intervention needed")
                    return False
                else:
                    logger.warning(f"⚠️ Unknown challenge choice: {choice}")
                    return False
            
            cl.challenge_code_handler = challenge_code_handler
            logger.info(f"✅ Challenge code handler set for {username} using email {email_address}")
        
        # Add delay to avoid rate limiting
        await asyncio.sleep(random.uniform(1, 3))
        
        # Check for stop event before login attempt
        if stop_event and stop_event.is_set():
            return {
                "username": username,
                "status": "cancelled",
                "message": "Login cancelled due to stop event"
            }
        
        # Attempt login - challenge_code_handler will automatically handle challenges
        # Use asyncio.to_thread to make the synchronous login call async
        try:
            await asyncio.to_thread(cl.login, username, password)
            logger.info(f"Login successful for {username}")
            
            # Ensure sessions directory exists before saving
            ensure_sessions_dir()
            
            # Save session (also make async)
            session_path = session_file_for(username)
            await asyncio.to_thread(cl.dump_settings, session_path)
            logger.info(f"Session saved for {username} to {session_path}")
            
            # Save successful login to verify.csv
            save_successful_login_to_verify(username, password, email_address, email_password)
            
            # Try to get user info asynchronously
            followers = None
            following = None
            ig_name = None
            bio = None
            instagram_id = None
            try:
                user_info = await get_user_info_from_session_async(cl, username)
                followers = user_info.get("followers")
                following = user_info.get("following")
                ig_name = user_info.get("ig_name")
                bio = user_info.get("bio")
                instagram_id = user_info.get("instagram_id")
            except Exception as e:
                logger.debug(f"Could not fetch user info for {username}: {e}")
            
            return {
                "username": username,
                "status": "success",
                "message": "Login successful",
                "session_path": session_path,
                "email": email_address,
                "email_password": email_password,
                "password": password,
                "proxy": proxy_to_use if proxy_to_use else None,
                "followers": followers,
                "following": following,
                "ig_name": ig_name,
                "bio": bio,
                "instagram_id": instagram_id,
            }
            
        except Exception as login_error:
            error_msg = str(login_error).lower()
            # Check if it's a challenge/verification error that resulted in suspension
            if "challenge" in error_msg or "verification" in error_msg or "suspended" in error_msg:
                logger.warning(f"🚫 Login failed for {username} - suspended due to no verification code")
                return {
                    "username": username,
                    "status": "suspended",
                    "message": "Login failed - no verification code received within 35 seconds",
                    "session_path": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                    "suspended_reason": "No email verification code received",
                }
            
            # Check for invalid password - mark as suspended
            if "bad_password" in error_msg or "invalid password" in error_msg or "wrong password" in error_msg:
                logger.warning(f"🚫 Invalid password for {username} - marking as suspended")
                return {
                    "username": username,
                    "status": "suspended",
                    "message": "Invalid password - login failed",
                    "session_path": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                    "suspended_reason": "Invalid password",
                }
            
            # Handle other specific error types
            if "challenge_required" in error_msg:
                return {
                    "username": username,
                    "status": "error",
                    "message": "Challenge required - manual verification needed",
                    "session_path": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                }
            elif "two_factor_required" in error_msg:
                return {
                    "username": username,
                    "status": "error",
                    "message": "Two-factor authentication required but no TOTP secret provided",
                    "session_path": None,
                    "email": email_address,
                    "email_password": email_password,
                    "password": password,
                }
            elif "invalid_user" in error_msg:
                return {
                    "username": username,
                    "status": "error",
                    "message": "Invalid username"
                }
            elif "rate_limit" in error_msg:
                return {
                    "username": username,
                    "status": "error",
                    "message": "Rate limited - try again later"
                }
            else:
                return {
                    "username": username,
                    "status": "error",
                    "message": f"Login failed: {str(login_error)}"
                }
                
    except Exception as e:
        logger.error(f"Error processing login for {username}: {e}")
        return {
            "username": username,
            "status": "error",
            "message": f"Processing error: {str(e)}"
        }


# Create FastAPI router
login_router = APIRouter()

# Stop event for cancelling batch operations
stop_events: Dict[str, threading.Event] = {}


def stop_all_operations():
    """
    Force stop all ongoing login operations.
    Called when Ctrl+C is pressed.
    """
    logger.warning("🛑 Force stopping all login operations...")
    for request_id, stop_event in stop_events.items():
        if stop_event and not stop_event.is_set():
            stop_event.set()
            logger.info(f"Stopped operation: {request_id}")
    
    # Clear all stop events after setting them
    stop_events.clear()
    logger.info("✅ All operations stopped.")


class AutomaticLoginData(BaseModel):
    thread_count: int


class LoginFromTextData(BaseModel):
    accounts: str  # Text containing accounts in format username:password:email:emailpassword (one per line)
    thread_count: int = 10
    
    
    class Config:
        json_schema_extra = {
            "example": {
                "accounts": "romola_98621_adelaida:Heritech@098A!2:bonnie8337412522@scalingtothemoon.com:Heritech@098A!2\nkerry_05211_hanni:Heritech@098A!2:mildred2892634948@scalingtothemoon.com:Heritech@098A!2",
                "thread_count": 10
            }
        }


@login_router.post("/automatic", tags=["Login"])
async def automatic_login(
    data: AutomaticLoginData,
):
    """
    Automatically login to Instagram accounts from verify.csv or verify.txt file.
    Supports two formats:
    1. CSV format: standard CSV with username, password, etc.
    2. TXT format: username:password:email:emailpassword (one per line)
       Example: stephenburrell848236:Test$20231704:max9951980523@scalingtothemoon.com:Heritech@098A!2
    
    Uses username and password only, retrieves email verification codes via IMAP if needed.
    """
    try:
        # Try to find CSV or TXT file
        # Get project root directory
        project_root = os.path.dirname(os.path.dirname(__file__))
        csv_path = os.path.join(project_root, "verify.csv")
        if not csv_path or not os.path.exists(csv_path):
            # Try TXT file if CSV not found
            txt_path = os.path.join(project_root, "verify.txt")
            if os.path.exists(txt_path):
                csv_path = txt_path
            else:
                raise HTTPException(status_code=400, detail=f"File not found: {csv_path or txt_path}")
        
        # Create stop event for this request
        request_id = str(uuid.uuid4())
        stop_event = threading.Event()
        stop_events[request_id] = stop_event
        
        logger.info(f"Starting automatic login from {csv_path} with {data.thread_count} threads")

        async def run_login_job():
            try:
                result = await batch_login_from_csv(csv_path, stop_event)
                result["request_id"] = request_id
                return result
            finally:
                # Clean up stop event mapping once finished
                stop_events.pop(request_id, None)

        # Always run in background (default behavior - not exposed in API)
        try:
            job = await background_tasks.create_task(
                name="login:automatic",
                coro_func=run_login_job,
                job_id=request_id,
            )
        except Exception:
            stop_events.pop(request_id, None)
            raise
        return {
            "job_id": job["id"],
            "status": job["status"],
            "request_id": request_id,
            "message": "Automatic login job queued",
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in automatic login: {e}")
        raise HTTPException(status_code=500, detail=f"Login failed: {str(e)}")


@login_router.post("/from-text", tags=["Login"])
async def login_from_text(
    data: LoginFromTextData,
):
    """
    Login to Instagram accounts from pasted text.
    
    Format: username:password:email:emailpassword (one per line, separated by colons)
    
    Examples:
    - romola_98621_adelaida:Heritech@098A!2:bonnie8337412522@scalingtothemoon.com:Heritech@098A!2
    - stephenburrell848236:Test$20231704:max9951980523@scalingtothemoon.com:Heritech@098A!2
    
    IMPORTANT: When pasting in Swagger UI, make sure to use valid JSON format.
    For multiline text in the 'accounts' field, newlines should be escaped as \\n
    OR use the /from-file endpoint to upload a .txt file directly.
    
    Uses username and password only, retrieves email verification codes via IMAP if needed.
    """
    tmp_path: Optional[str] = None
    request_id: Optional[str] = None
    try:
        # Create a temporary file with the pasted text
        with tempfile.NamedTemporaryFile(mode='w', suffix='.txt', delete=False, encoding='utf-8') as tmp_file:
            # Write the accounts text (handles multiple accounts separated by newlines)
            # Pydantic automatically converts escaped \n to actual newlines
            tmp_file.write(data.accounts)
            tmp_path = tmp_file.name
        
        # Create stop event for this request
        request_id = str(uuid.uuid4())
        stop_event = threading.Event()
        stop_events[request_id] = stop_event
        
        logger.info(f"Starting login from pasted text with {data.thread_count} threads")
        logger.info(f"Parsing {len(data.accounts.splitlines())} lines of account data")

        async def run_login_job():
            try:
                result = await batch_login_from_csv(tmp_path, stop_event)
                result["request_id"] = request_id
                return result
            finally:
                stop_events.pop(request_id, None)
                if tmp_path and os.path.exists(tmp_path):
                    os.unlink(tmp_path)

        # Always run in background (default behavior - not exposed in API)
        try:
            job = await background_tasks.create_task(
                name="login:from-text",
                coro_func=run_login_job,
                job_id=request_id,
            )
        except Exception:
            stop_events.pop(request_id, None)
            if tmp_path and os.path.exists(tmp_path):
                os.unlink(tmp_path)
            raise
        return {
            "job_id": job["id"],
            "status": job["status"],
            "request_id": request_id,
            "message": "Login from text job queued",
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in login from text: {e}")
        if tmp_path and os.path.exists(tmp_path):
            os.unlink(tmp_path)
        if request_id:
            stop_events.pop(request_id, None)
        raise HTTPException(status_code=500, detail=f"Login failed: {str(e)}")


@login_router.post("/from-file", tags=["Login"])
async def login_from_file(
    file: UploadFile = File(...),
    thread_count: int = Form(10),
):
    """
    Login to Instagram accounts from uploaded .txt file.
    
    Format: username:password:email:emailpassword (one per line)
    Example: stephenburrell848236:Test$20231704:max9951980523@scalingtothemoon.com:Heritech@098A!2
    
    Upload a .txt file with accounts, one per line.
    Uses username and password only, retrieves email verification codes via IMAP if needed.
    """
    tmp_path: Optional[str] = None
    request_id: Optional[str] = None
    try:
        # Check file extension
        if not file.filename or not file.filename.endswith('.txt'):
            raise HTTPException(status_code=400, detail="Only .txt files are supported")
        
        # Create a temporary file to save uploaded content
        with tempfile.NamedTemporaryFile(mode='wb', suffix='.txt', delete=False) as tmp_file:
            content = await file.read()
            tmp_file.write(content)
            tmp_path = tmp_file.name
        
        # Create stop event for this request
        request_id = str(uuid.uuid4())
        stop_event = threading.Event()
        stop_events[request_id] = stop_event
        
        logger.info(f"Starting login from uploaded file {file.filename} with {thread_count} threads")

        async def run_login_job():
            try:
                result = await batch_login_from_csv(tmp_path, stop_event)
                result["request_id"] = request_id
                return result
            finally:
                stop_events.pop(request_id, None)
                if tmp_path and os.path.exists(tmp_path):
                    os.unlink(tmp_path)

        # Always run in background (default behavior - not exposed in API)
        try:
            job = await background_tasks.create_task(
                name="login:from-file",
                coro_func=run_login_job,
                job_id=request_id,
            )
        except Exception:
            stop_events.pop(request_id, None)
            if tmp_path and os.path.exists(tmp_path):
                os.unlink(tmp_path)
            raise
        return {
            "job_id": job["id"],
            "status": job["status"],
            "request_id": request_id,
            "message": "Login from file job queued",
        }
        
    except HTTPException:
        raise
    except Exception as e:
        logger.error(f"Error in login from file: {e}")
        if tmp_path and os.path.exists(tmp_path):
            os.unlink(tmp_path)
        if request_id:
            stop_events.pop(request_id, None)
        raise HTTPException(status_code=500, detail=f"Login failed: {str(e)}")


@login_router.post("/cancel/{request_id}", tags=["Login"])
async def cancel_login(request_id: str):
    """
    Cancel an ongoing batch login operation.
    """
    if request_id in stop_events:
        stop_events[request_id].set()
        return {"status": "ok", "message": "Login cancellation requested"}
    else:
        raise HTTPException(status_code=404, detail="Request ID not found")
