import os
import random
import requests
from requests.exceptions import RequestException
from concurrent.futures import ThreadPoolExecutor
from bs4 import BeautifulSoup
from datetime import datetime
from fastapi import APIRouter, File, UploadFile, HTTPException
from fastapi.responses import JSONResponse
from typing import List, Optional
import tempfile
import asyncio
import aiohttp
import aiofiles

# Create router for username checking
router = APIRouter(prefix="/check", tags=["Username Checker"])

# Semaphore to limit concurrent requests to 5
request_semaphore = asyncio.Semaphore(5)

# Proxy configuration - load from proxies.txt file
PROXIES_TXT = os.path.join(os.path.dirname(os.path.dirname(__file__)), "proxies.txt")
_proxies_cache: Optional[List[str]] = None

def load_proxies_from_file() -> List[str]:
    """
    Load proxies from proxies.txt file.
    Each line should contain one proxy in format: scheme://user:pass@host:port
    Returns empty list if file doesn't exist or is empty.
    """
    global _proxies_cache
    
    # Return cached proxies if available
    if _proxies_cache is not None:
        return _proxies_cache
    
    proxies = []
    if os.path.exists(PROXIES_TXT):
        try:
            with open(PROXIES_TXT, 'r', encoding='utf-8') as f:
                for line in f:
                    line = line.strip()
                    # Skip empty lines and comments
                    if line and not line.startswith('#'):
                        proxies.append(line)
            print(f"✅ Loaded {len(proxies)} proxies from {PROXIES_TXT}")
        except Exception as e:
            print(f"Error loading proxies from {PROXIES_TXT}: {e}")
    else:
        print(f"⚠️ Proxies file not found: {PROXIES_TXT}")
    
    # Cache the proxies
    _proxies_cache = proxies
    return proxies

def get_random_proxy() -> Optional[str]:
    """
    Get a random proxy from proxies.txt file.
    Returns None if no proxies are available.
    Ensures proxy has proper scheme (http:// or https://).
    """
    proxies = load_proxies_from_file()
    if proxies:
        proxy = random.choice(proxies)
        # Ensure proxy has scheme
        if proxy and not proxy.startswith(('http://', 'https://', 'socks5://', 'socks4://')):
            # Add http:// if no scheme
            proxy = f"http://{proxy}"
        return proxy
    return None

def get_proxies_dict() -> Optional[dict]:
    """
    Get a random proxy as a dict for requests library.
    Returns None if no proxies are available.
    """
    proxy_url = get_random_proxy()
    if proxy_url:
        return {
            "http": proxy_url,
            "https": proxy_url
        }
    return None


def read_usernames_from_file(filename):
    """Read usernames from a text file with format username:password:email:password."""
    if not os.path.exists(filename):
        print(f"File {filename} not found.")
        return []
    
    try:
        with open(filename, 'r', encoding='utf-8') as file:
            usernames = []
            for line in file:
                line = line.strip()
                if line:
                    # Extract username (part before the first colon)
                    username = line.split(':')[0]
                    if username:
                        usernames.append(username)
        return usernames
    except Exception as e:
        print(f"Error reading file {filename}: {e}")
        return []

def read_usernames_from_content(content: str):
    """Read usernames from file content with format username:password:email:password."""
    try:
        usernames = []
        lines = content.split('\n')
        for line in lines:
            line = line.strip()
            if line:
                # Extract username (part before the first colon)
                username = line.split(':')[0]
                if username:
                    usernames.append(username)
        return usernames
    except Exception as e:
        print(f"Error reading file content: {e}")
        return []

def write_usernames_to_file(filename, usernames):
    """Write usernames to a text file, one per line."""
    try:
        with open(filename, 'w', encoding='utf-8') as file:
            for username in usernames:
                file.write(f"{username}\n")
        print(f"Written {len(usernames)} usernames to {filename}")
    except Exception as e:
        print(f"Error writing to file {filename}: {e}")

def calculate_statistics(total_count, active_count, inactive_count):
    """Calculate and return statistics with percentages."""
    if total_count == 0:
        return {
            "total": 0,
            "active": 0,
            "inactive": 0,
            "active_percentage": 0.0,
            "inactive_percentage": 0.0
        }
    
    active_percentage = (active_count / total_count) * 100
    inactive_percentage = (inactive_count / total_count) * 100
    
    return {
        "total": total_count,
        "active": active_count,
        "inactive": inactive_count,
        "active_percentage": round(active_percentage, 2),
        "inactive_percentage": round(inactive_percentage, 2)
    }


async def check_instagram_account_async(username, session):
    """Check if the Instagram account is active by sending an async request."""
    url = f"https://www.instagram.com/{username}/"
    
    try:
        # Get random proxy from proxies.txt
        proxy_url = get_random_proxy()
        
        # Send an async request with proxy (if available)
        async with session.get(url, proxy=proxy_url, timeout=30) as response:
            if response.status == 200:
                # Parse the page content
                content = await response.text()
                soup = BeautifulSoup(content, 'html.parser')

                # Check for elements that indicate an active profile
                if soup.find('meta', property="og:image") and soup.find('meta', property="og:description"):
                    return True
                else:
                    return False
            elif response.status == 404:
                # Account not found (could be deleted, suspended, etc.)
                return False
            else:
                # Other errors (e.g., blocked by Instagram)
                return None
    except Exception as e:
        print(f"An error occurred while checking {username}: {e}")
        return None

async def is_instagram_account_active_async(username, session):
    """Check if the Instagram account is active (using async requests)."""
    account_status = await check_instagram_account_async(username, session)
    
    if account_status is True:
        print(f"Account {username} is active.")
        return True
    elif account_status is False:
        print(f"Account {username} is not found or has been suspended.")
        return False
    else:
        print(f"Could not determine the status of account {username}.")
        return False

# Keep the original sync functions for backward compatibility
def check_instagram_account(username, proxies_dict=None):
    """Check if the Instagram account is active by sending a request."""
    url = f"https://www.instagram.com/{username}/"
    
    try:
        # Get random proxy from proxies.txt if not provided
        if proxies_dict is None:
            proxies_dict = get_proxies_dict()
        
        # Send a request with a proxy (if available)
        response = requests.get(url, proxies=proxies_dict, timeout=30)
        
        if response.status_code == 200:
            # Parse the page content
            soup = BeautifulSoup(response.text, 'html.parser')

            # Check for elements that indicate an active profile
            if soup.find('meta', property="og:image") and soup.find('meta', property="og:description"):
                return True
            else:
                return False
        elif response.status_code == 404:
            # Account not found (could be deleted, suspended, etc.)
            return False
        else:
            # Other errors (e.g., blocked by Instagram)
            return None
    except RequestException as e:
        print(f"An error occurred while checking {username}: {e}")
        return None

def is_instagram_account_active(username, proxies_dict=None):
    """Check if the Instagram account is active (using requests)."""
    account_status = check_instagram_account(username, proxies_dict)
    
    if account_status is True:
        print(f"Account {username} is active.")
        return True
    elif account_status is False:
        print(f"Account {username} is not found or has been suspended.")
        return False
    else:
        print(f"Could not determine the status of account {username}.")
        return False




def run_username_check():
    """Standalone function to check usernames from check.txt file."""
    # Read usernames from check.txt
    usernames = read_usernames_from_file("check.txt")
    
    if not usernames:
        print("No usernames found in check.txt file.")
        return
    
    print(f"Starting to check {len(usernames)} usernames...")
    
    active_accounts = []
    inactive_accounts = []

    # Use ThreadPoolExecutor to handle the checks in parallel
    with ThreadPoolExecutor(max_workers=50) as executor:
        # Map the usernames to the check function for parallel processing
        # Each check will get a random proxy from proxies.txt
        results = list(executor.map(lambda username: (username, is_instagram_account_active(username)), usernames))

    # Process the results
    for username, is_active in results:
        if is_active:
            active_accounts.append(username)
        else:
            inactive_accounts.append(username)
    
    # Write results to files
    write_usernames_to_file("active.txt", active_accounts)
    write_usernames_to_file("non-active.txt", inactive_accounts)
    
    # Calculate statistics
    stats = calculate_statistics(len(usernames), len(active_accounts), len(inactive_accounts))
    
    # Print statistics to console
    print(f"\n=== STATISTICS ===")
    print(f"Total accounts checked: {stats['total']}")
    print(f"Active accounts: {stats['active']} ({stats['active_percentage']}%)")
    print(f"Inactive accounts: {stats['inactive']} ({stats['inactive_percentage']}%)")
    print(f"Active accounts saved to: active.txt")
    print(f"Inactive accounts saved to: non-active.txt")
    print(f"Check completed at: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    
    return stats

@router.get("/")
async def root():
    """Root endpoint with API information."""
    return {
        "message": "Instagram Username Checker API",
        "version": "1.0.0",
        "proxy": "Loaded from proxies.txt (random selection)",
        "concurrent_requests_limit": 5,
        "endpoints": {
            "POST /check/check-usernames": "Upload a text file to check Instagram usernames (results in response only)",
            "POST /check/check-usernames-with-files": "Upload a text file to check Instagram usernames (saves to unique timestamped files)",
            "GET /check/": "API information"
        },
        "note": "Each concurrent request gets its own results - no file conflicts"
    }

@router.post("/check-usernames-with-files")
async def check_usernames_with_files_api(file: UploadFile = File(...)):
    """Upload a text file and check Instagram usernames, save results to unique files."""
    
    # Acquire semaphore to limit concurrent requests
    async with request_semaphore:
        # Validate file type
        if not file.filename.endswith('.txt'):
            raise HTTPException(status_code=400, detail="Only .txt files are allowed")
        
        try:
            # Read file content
            content = await file.read()
            content_str = content.decode('utf-8')
            
            # Extract usernames from content
            usernames = read_usernames_from_content(content_str)
            
            if not usernames:
                raise HTTPException(status_code=400, detail="No valid usernames found in the file")
            
            print(f"Starting to check {len(usernames)} usernames from uploaded file...")
            
            active_accounts = []
            inactive_accounts = []

            # Create async HTTP session
            connector = aiohttp.TCPConnector(limit=100, limit_per_host=30)
            timeout = aiohttp.ClientTimeout(total=30)
            
            async with aiohttp.ClientSession(connector=connector, timeout=timeout) as session:
                # Create tasks for all username checks
                tasks = []
                for username in usernames:
                    task = is_instagram_account_active_async(username, session)
                    tasks.append((username, task))
                
                # Execute all tasks concurrently using asyncio.gather
                task_results = await asyncio.gather(*[task for _, task in tasks], return_exceptions=True)
                
                # Process results
                results = []
                for i, (username, _) in enumerate(tasks):
                    if isinstance(task_results[i], Exception):
                        print(f"Error checking {username}: {task_results[i]}")
                        results.append((username, False))
                    else:
                        results.append((username, task_results[i]))

            # Process the results
            for username, is_active in results:
                if is_active:
                    active_accounts.append(username)
                else:
                    inactive_accounts.append(username)
            
            # Calculate statistics
            stats = calculate_statistics(len(usernames), len(active_accounts), len(inactive_accounts))
            
            # Create unique filenames with timestamp
            timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
            active_filename = f"active_{timestamp}.txt"
            inactive_filename = f"non-active_{timestamp}.txt"
            
            # Write results to unique files
            write_usernames_to_file(active_filename, active_accounts)
            write_usernames_to_file(inactive_filename, inactive_accounts)
            
            # Return results in response
            return {
                "message": "Username checking completed successfully",
                "filename": file.filename,
                "statistics": stats,
                "active_accounts": active_accounts,
                "inactive_accounts": inactive_accounts,
                "processed_at": datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                "saved_files": {
                    "active_file": active_filename,
                    "inactive_file": inactive_filename
                }
            }
            
        except UnicodeDecodeError:
            raise HTTPException(status_code=400, detail="File encoding error. Please ensure the file is UTF-8 encoded")
        except Exception as e:
            raise HTTPException(status_code=500, detail=f"Error processing file: {str(e)}")

@router.post("/check-usernames")
async def check_usernames_api(file: UploadFile = File(...)):
    """Upload a text file and check Instagram usernames."""
    
    # Acquire semaphore to limit concurrent requests
    async with request_semaphore:
        # Validate file type
        if not file.filename.endswith('.txt'):
            raise HTTPException(status_code=400, detail="Only .txt files are allowed")
        
        try:
            # Read file content
            content = await file.read()
            content_str = content.decode('utf-8')
            
            # Extract usernames from content
            usernames = read_usernames_from_content(content_str)
            
            if not usernames:
                raise HTTPException(status_code=400, detail="No valid usernames found in the file")
            
            print(f"Starting to check {len(usernames)} usernames from uploaded file...")
            
            active_accounts = []
            inactive_accounts = []

            # Create async HTTP session
            connector = aiohttp.TCPConnector(limit=100, limit_per_host=30)
            timeout = aiohttp.ClientTimeout(total=30)
            
            async with aiohttp.ClientSession(connector=connector, timeout=timeout) as session:
                # Create tasks for all username checks
                tasks = []
                for username in usernames:
                    task = is_instagram_account_active_async(username, session)
                    tasks.append((username, task))
                
                # Execute all tasks concurrently using asyncio.gather
                task_results = await asyncio.gather(*[task for _, task in tasks], return_exceptions=True)
                
                # Process results
                results = []
                for i, (username, _) in enumerate(tasks):
                    if isinstance(task_results[i], Exception):
                        print(f"Error checking {username}: {task_results[i]}")
                        results.append((username, False))
                    else:
                        results.append((username, task_results[i]))

            # Process the results
            for username, is_active in results:
                if is_active:
                    active_accounts.append(username)
                else:
                    inactive_accounts.append(username)
            
            # Calculate statistics
            stats = calculate_statistics(len(usernames), len(active_accounts), len(inactive_accounts))
            
            # Return results in response (no file saving to avoid conflicts)
            return {
                "message": "Username checking completed successfully",
                "filename": file.filename,
                "statistics": stats,
                "active_accounts": active_accounts,
                "inactive_accounts": inactive_accounts,
                "processed_at": datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                "note": "Results returned in response only - no files saved to avoid conflicts with concurrent requests"
            }
            
        except UnicodeDecodeError:
            raise HTTPException(status_code=400, detail="File encoding error. Please ensure the file is UTF-8 encoded")
        except Exception as e:
            raise HTTPException(status_code=500, detail=f"Error processing file: {str(e)}")

# Router will be imported in app.py
# No need to run server here - app.py will handle that
